<?php
/**
 * Licensed under The GPL-3.0 License
 * For full copyright and license information, please see the LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @since    2.0.0
 * @author   Christopher Castro <chris@quickapps.es>
 * @link     http://www.quickappscms.org
 * @license  http://opensource.org/licenses/gpl-3.0.html GPL-3.0 License
 */
namespace Node\Model\Entity;

use Cake\Error\FatalErrorException;
use Cake\I18n\I18n;
use Cake\ORM\Entity;
use Cake\ORM\TableRegistry;
use Cake\Routing\Router;
use Node\Model\Entity\NodeType;
use User\Model\Entity\AccessibleEntityTrait;
use User\Model\Entity\User;

/**
 * Represents a single "node" within "nodes" table.
 *
 * @property int $id
 * @property int $node_type_id
 * @property int $translation_for
 * @property int $promote
 * @property int $sticky
 * @property int $comment_status
 * @property int $status
 * @property int $created_by
 * @property int $modified_by
 * @property string $node_type_slug
 * @property string $slug
 * @property string $title
 * @property string $description
 * @property string $language
 * @property string $type
 * @property string $url
 * @property array $roles
 * @property \User\Model\Entity\User $author
 * @method bool isAccessible(array|null $roles = null)
 */
class Node extends Entity
{

    use AccessibleEntityTrait;

    /**
     * Gets node type.
     *
     * As Node Types are not dependent of Nodes (deleting a node_type won't remove
     * all nodes of that type). Some types we found nodes without `node_type`, in
     * that cases, if no node_type is found `--unknow--` will be returned.
     *
     * @return string
     */
    protected function _getType()
    {
        $name = __d('node', '(unknown)');
        if ($this->has('node_type') && $this->get('node_type')->has('name')) {
            $name = $this->get('node_type')->get('name');
        }
        return $name;
    }

    /**
     * Gets node's details page URL.
     *
     * Node's details URL's follows the syntax below:
     *
     *     http://example.com/[node-type-slug]/[node-slug].html
     *
     * Example:
     *
     *     http://example.com/blog-article/my-first-article.html
     *
     * @return string
     */
    protected function _getUrl()
    {
        $url = Router::getRequest()->base;
        if (option('url_locale_prefix')) {
            $url .= '/' . I18n::locale();
        }

        $url .= "/{$this->node_type_slug}/{$this->slug}.html";
        return Router::normalize($url);
    }

    /**
     * Gets node's author as an User entity.
     *
     * @return \User\Model\Entity\User
     */
    protected function _getAuthor()
    {
        if ($this->has('user')) {
            return $this->get('user');
        } elseif (!empty($this->created_by)) {
            $user = TableRegistry::get('User.Users')
                ->find()
                ->where(['id' => $this->created_by])
                ->first();

            if ($user) {
                return $user;
            }
        }

        return new User([
            'username' => __d('node', 'unknown'),
            'name' => __d('node', 'Unknown'),
            'web' => __d('node', '(no website)'),
            'email' => __d('node', 'Unknown'),
        ]);
    }

    /**
     * Gets the parent node for which this node is a translation of.
     *
     * @return mixed The parent node if exists, null otherwise
     */
    public function parentLocale()
    {
        if (!$this->has('slug')) {
            throw new FatalErrorException(__d('node', "Missing property 'slug', make sure to include it using Query::select()."));
        }

        return TableRegistry::get('Node.Nodes')
            ->find()
            ->select(['id', 'slug', 'node_type_slug', 'language'])
            ->where([
                'slug' => $this->slug,
                'status' => 1,
                'translation_for NOT IN' => ['', null]
            ])
            ->first();
    }

    /**
     * Find if this node has a translation to the given locale code.
     *
     * @param string|null $locale Locale code for which look for translations,
     *  if not given current language code will be used
     * @return mixed Translation entity if exists, null otherwise
     * @throws Cake\Error\FatalErrorException When if any of the required
     *  properties is not present in this entity
     */
    public function hasTranslation($locale = null)
    {
        if (!$this->has('id') || !$this->has('node_type_slug')) {
            throw new FatalErrorException(__d('node', "Missing properties 'id' or 'node_type_slug', make sure to include them using Query::select()."));
        }

        if ($locale === null) {
            $locale = I18n::locale();
        }

        return TableRegistry::get('Node.Nodes')
            ->find()
            ->select(['id', 'slug', 'node_type_slug', 'language'])
            ->where([
                'translation_for' => $this->id,
                'node_type_slug' => $this->node_type_slug,
                'language' => $locale,
                'status' => 1,
            ])
            ->first();
    }

    /**
     * Set defaults content settings based on parent content type.
     *
     * You can provide a NodeType entity to fetch defaults values. By default if
     * none is provided it automatically fetches the information from the
     * corresponding Content Type.
     *
     * @param bool|\Node\Model\Entity\NodeType $type False for auto fetch or a
     *  NodeType entity to extract information from
     * @return void
     * @throws Cake\Error\FatalErrorException When content type was not found for
     *  this content node.
     */
    public function setDefaults($type = false)
    {
        if (!$type) {
            if (!$this->has('node_type_slug') && !$this->has('id')) {
                throw new FatalErrorException(__d('node', 'Unable to get Content-Type information.'));
            }

            if (!$this->has('node_type_slug')) {
                $nodeTypeSlug = TableRegistry::get('Node.Nodes')->find()
                    ->select(['node_type_slug'])
                    ->where(['id' => $this->get('id')])
                    ->first();
                $nodeTypeSlug = $nodeTypeSlug->node_type_slug;
            } else {
                $nodeTypeSlug = $this->get('node_type_slug');
            }

            $type = TableRegistry::get('Node.NodeTypes')->find()
                ->where(['slug' => $nodeTypeSlug])
                ->first();
        }

        if (!($type instanceof NodeType) || !$type->has('defaults')) {
            throw new FatalErrorException(__d('node', "Node::setDefaults() was unable to get Content Type defaults values."));
        }

        $this->set('language', $type->defaults['language']);
        $this->set('comment_status', $type->defaults['comment_status']);
        $this->set('status', $type->defaults['status']);
        $this->set('promote', $type->defaults['promote']);
        $this->set('sticky', $type->defaults['sticky']);
    }
}
