<?php

declare(strict_types=1);

/*
 * This file is part of the Jejik\MT940 library
 *
 * Copyright (c) 2020 Powercloud GmbH <d.richter@powercloud.de>
 * Licensed under the MIT license
 *
 * For the full copyright and license information, please see the LICENSE
 * file that was distributed with this source code.
 */

namespace Jejik\MT940\Parser;

use Jejik\MT940\Balance;
use Jejik\MT940\BalanceInterface;

/**
 * Lbbw provides a parser for Lbbw Bank
 * @package Jejik\MT940\Parser
 */
class Lbbw extends GermanBank
{
    /**
     * Test if the document can be read by the parser
     */
    public function accept(string $text): bool
    {
        $allowedUniqueIdentifiers = [
            ':20:LBBW'
        ];

        // unique identifier check
        $mt940Identifier = substr($text, 0, 8);
        if (in_array($mt940Identifier, $allowedUniqueIdentifiers)) {
            return true;
        }

        // if not check it's BLZ
        return $this->isBLZAllowed($text);
    }

    /**
     * Get an array of allowed BLZ for this bank
     */
    public function getAllowedBLZ(): array
    {
        return [
            '60050101'
        ];
    }

    /**
     * Get the contra account from a transaction
     *
     * @param array $lines The transaction text at offset 0 and the description at offset 1
     */
    protected function contraAccountNumber(array $lines): ?string
    {
        if (preg_match('/\?31(.*?)\?32/s', $lines[1], $match)) {
            return trim(preg_replace('/\s\s+/', '', $match[1]));
        }

        return null;
    }

    protected function contraAccountName(array $lines): ?string
    {
        if (preg_match('/\?32(.*?)\?34/s', $lines[1], $match)) {
            return trim(preg_replace('/\s\s+/', '', $match[1]));
        }

        return null;
    }

    //TODO: reformat :86: Mehrzweckfeld
    protected function description(?string $description): ?string
    {
        return parent::description($description); // TODO: Change the autogenerated stub
    }
    /**
     * This bank also has 4 characters for the currency in the balance line which must be read out.
     *
     * @param \Jejik\MT940\BalanceInterface $balance
     * @param string $text
     * @return \Jejik\MT940\BalanceInterface
     */
    protected function balance(\Jejik\MT940\BalanceInterface $balance, string $text): \Jejik\MT940\BalanceInterface
    {
        if (!preg_match('/(C|D)(\d{6})([A-Z]{3,4})([0-9,]{1,15})/', $text, $match)) {
            throw new \RuntimeException(sprintf('Cannot parse balance: "%s"', $text));
        }

        $amount = (float)str_replace(',', '.', $match[4]);
        if ($match[1] === 'D') {
            $amount *= -1;
        }

        $date = \DateTime::createFromFormat('ymd', $match[2]);
        $date->setTime(0, 0, 0);

        $balance
            ->setCurrency($match[3])
            ->setAmount($amount)
            ->setDate($date);

        return $balance;
    }

    /**
     * Get the closing balance
     */
    protected function closingBalance(string $text): ?Balance
    {
        if ($line = $this->getLine('62F|62M', $text)) {
            return $this->balance($this->reader->createClosingBalance(), $line);
        }

        if ($line = $this->getClosingLineWithoutColin('62F|62M', $text)) {
            return $this->balance($this->reader->createClosingBalance(), $line);
        }

        return null;
    }

    /**
     * This method determines the closing balance when there is no colon at the beginning of the closing balance line to
     * identify and separate (usually it is :<id>: but this bank can have :<id>xxx)
     *
     * @param string $id
     * @param string $text
     * @param int $offset
     * @param int|null $position
     * @param int|null $length
     * @return string|null
     */
    private function getClosingLineWithoutColin(
        string $id,
        string $text,
        int $offset = 0,
        int &$position = null,
        int &$length = null
    ): ?string {
        $pcre = '/(?:^|\r\n)\:(' . $id . ')'   // ":<id>" at the start of a line
            . '(.+)'                           // Contents of the line
            . '(:?$|\r\n\:[[:alnum:]]{2,3}\:)' // End of the text or next ":<id>:"
            . '/Us';                           // Ungreedy matching

        $substring = substr($text, $offset);
        if ($substring !== false) {
            // Offset manually, so the start of the offset can match ^
            if (preg_match($pcre, $substring, $match, PREG_OFFSET_CAPTURE)) {
                $position = $offset + $match[1][1] - 1;
                $length = strlen($match[2][0]);

                return rtrim($match[2][0]);
            }
        }
        return null;
    }
}
