/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { ComponentEventMap } from "./Component.js";
/**
 * Component events
 *
 * @see ComponentEventMap
 */
export interface ObservableEventMap<Type> {
}
/**
 * Observable listener configuration.
 *
 * A listener can be passed as a function or a config option.
 *
 * {@see ObservableListenerWithOpts}
 * {@see ObservableConfig.listeners}
 */
export type ObservableListener<Map extends ObservableEventMap<Observable>> = {
    [P in keyof Map]?: ObservableListenerWithOpts<Map[P]> | Map[P];
};
export type Listener<Comp extends Observable = Observable> = (sender: Comp, ...args: any[]) => false | void;
/**
 * Options for event listener functions
 */
export interface ObservableListenerOpts {
    /**
     * Fire the listener only once
     */
    once?: boolean;
    /**
     * Buffer the listener with this number of milliseconds.
     * When an event is fired multiple times within the buffer period only the last event listener will be called
     */
    buffer?: number;
    /**
     * Delay the listener with this number of milliseconds.
     */
    delay?: number;
    /**
     * Put this listener before existing listeners instead of after
     */
    unshift?: boolean;
}
/**
 * The listener with options can be passed to a component config object's listeners property.
 */
interface ObservableListenerWithOpts<fn> extends ObservableListenerOpts {
    /**
     * The listener function to execute on the event
     */
    fn: fn;
}
/**
 * Observable
 *
 * Adds event listener functionality
 */
export declare class Observable {
    private lisnrs;
    /**
     * Add a listener
     *
     * @param eventName
     * @param listener
     * @param options
     */
    on<key extends keyof ObservableEventMap<this>, L extends Listener>(eventName: keyof ObservableEventMap<this>, listener: ObservableEventMap<this>[key], options?: ObservableListenerOpts): L;
    /**
     * Override this function to lazily initialize events
     * @param eventName
     * @protected
     */
    protected onFirstListenerAdded(eventName: string): void;
    private once;
    /**
     * Remove listener
     *
     * @param eventName
     * @param listener
     */
    un(eventName: keyof ObservableEventMap<this>, listener: Function): boolean;
    /**
     * Fire an event
     *
     * When a listener returns false this function will return false too.
     *
     * @param eventName
     * @param args
     */
    fire<K extends keyof ObservableEventMap<this>>(eventName: K, ...args: Parameters<ObservableEventMap<this>[K]>): boolean;
    protected relayEvent(comp: Observable, type: any): void;
}
type Writeable<T> = {
    -readonly [P in keyof T]: T[P];
};
type CompFuncs = "buildState" | "cascade" | "computeZIndex" | "createFindPredicateFunction" | "findAncestor" | "findAncestorByType" | "findChild" | "findChildByType" | "findChildren" | "findChildrenByType" | "findItem" | "findItemIndex" | "fire" | "focus" | "getInsertBefore" | "getState" | "hasOwnProperty" | "hasState" | "hide" | "initClassName" | "initItems" | "internalRemove" | "internalRender" | "internalSetHidden" | "isFocusable" | "mask" | "nextSibling" | "on" | "onFirstListenerAdded" | "once" | "previousSibling" | "print" | "propertyIsEnumerable" | "relayEvent" | "remove" | "render" | "renderItem" | "renderItems" | "restoreState" | "saveState" | "show" | "toLocaleString" | "toString" | "un" | "unmask" | "valueOf" | "items";
/**
 * Generic Config option that allows all public properties as options.
 * It excludes all function types. If you need to pass functions as config options you will need to add them like this:
 *
 * ```
 * type ChipsConfig = Config<ChipsField, FieldEventMap<ChipsField>> &
 *   // Add the function properties as they are filtered out
 * 	Partial<Pick<ChipsField, "textInputToValue" | "chipRenderer">>
 * ```
 */
export type Config<Cmp extends Observable, EventMap extends ObservableEventMap<Observable> = ComponentEventMap<Cmp>, Required extends keyof Cmp = never, OmittedProps extends keyof Cmp = never> = Writeable<Partial<Omit<Cmp, CompFuncs>>> & Writeable<Pick<Cmp, Required>> & {
    /**
     * Add listeners
     *
     * You can pass them in two ways:
     *
     * @example
     * ```
     * comp({
     * 		listeners: {
     *
     * 			//simple syntax add the listener directly
     * 			additem:(container, item, index) => {
     * 				//do something when an item was added
     * 			},
     *
     * 	    	// extended syntax to pass extra listener options
     * 			beforeadditem: {
     * 				//with this syntax you can pass extra options
     * 				fn:(container, item, index) => {
     * 					//do something before an item will be added but only once
     * 				},
     * 				once: true
     * 			}
     * 		}
     * 	})
     * 	```
     *
     * 	@see Observable.on()
     */
    listeners?: ObservableListener<EventMap>;
};
export {};
//# sourceMappingURL=Observable.d.ts.map