/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { btn, Button } from "./Button.js";
import { tbar } from "./Toolbar.js";
import { t } from "../Translate.js";
import { textfield } from "./form/TextField.js";
import { createComponent } from "./Component.js";
import { FunctionUtil } from "../util/FunctionUtil.js";
export class SearchButton extends Button {
    constructor() {
        super();
        this.buffer = 300;
        this.handler = (button, ev) => {
            this.mainTbar = button.parent;
            this.getSearchTBar().show();
            this.searchField.focus();
        };
        this.icon = "search";
        this.title = t("Search");
        this.searchField = textfield({
            label: t("Search"),
            flex: 1,
            buttons: [
                btn({
                    type: "button",
                    icon: "clear",
                    handler: () => {
                        this.reset();
                    }
                })
            ]
        });
        this.searchField.on("render", () => {
            this.searchField.input.addEventListener('input', FunctionUtil.buffer(this.buffer, this.onInput.bind(this)));
            this.searchField.el.addEventListener('keydown', (e) => {
                if (e.key == "Enter") {
                    e.preventDefault();
                    e.stopPropagation();
                }
                if (e.key == "Escape") {
                    e.preventDefault();
                    e.stopPropagation();
                    this.close();
                }
            });
        });
    }
    reset() {
        this.searchField.reset();
        this.close();
        this.fire("reset", this);
        this.fire("input", this, "");
        this.el.classList.remove("accent");
        this.el.classList.remove("filled");
    }
    close() {
        this.searchTBar.hide();
        this.mainTbar.show();
        this.focus();
    }
    onInput() {
        this.el.classList.toggle("accent", !!this.searchField.value);
        this.el.classList.toggle("filled", !!this.searchField.value);
        this.fire("input", this, this.searchField.value);
    }
    getSearchTBar() {
        if (!this.searchTBar) {
            this.searchTBar = tbar({
                cls: "search"
            }, btn({
                icon: "chevron_left",
                title: t("Back"),
                handler: () => {
                    this.close();
                }
            }), this.searchField);
            this.mainTbar.items.add(this.searchTBar);
        }
        return this.searchTBar;
    }
}
/**
 * Shorthand function to create {@see SearchButton}
 *
 * @example
 * ```
 * searchbtn({
 * 	listeners: {
 * 		input:(searchBtn, text) => {
 *
 * 			const filtered = records.filter((r) => {
 * 				return !text || r.description.toLowerCase().indexOf(text.toLowerCase()) === 0;
 * 			});
 *
 * 			//simple local filter on the store
 * 			table.store.loadData(filtered, false)
 * 		}
 * 	}
 * })
 * ```
 *
 * @param config
 * @param items
 */
export const searchbtn = (config) => createComponent(new SearchButton(), config);
//# sourceMappingURL=SearchButton.js.map