/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Field } from "./Field.js";
import { createComponent } from "../Component.js";
/**
 * Field that contains fields.
 *
 * The value that it returns is an object with the field names as keys.
 */
export class ContainerField extends Field {
    constructor(tagName = "div") {
        super(tagName);
        this.baseCls = "";
        this.hideLabel = true;
        this.fireChangeOnBlur = false;
        /**
         * Not needed on container field as the fields within handle this.
         */
        this.validateOnBlur = false;
        this.cls = "flow";
    }
    /**
     * Find all form fields
     *
     * @param nameOrItemId If given only items with matching name or itemId are returned
     */
    findFields(nameOrItemId = undefined) {
        const fields = [];
        const fn = (item) => {
            if (item == this) {
                return;
            }
            if (item.isFormField) {
                if (!nameOrItemId || (item.name == nameOrItemId || item.itemId == nameOrItemId)) {
                    fields.push(item);
                }
                return false;
            }
        };
        this.cascade(fn);
        return fields;
    }
    /**
     * Find form field by name or item ID
     *
     * It cascades down the component hierarchy.
     *
     * @param nameOrItemId
     */
    findField(nameOrItemId) {
        let field;
        const fn = (item) => {
            if ((item.isFormField) && item.name == nameOrItemId || item.itemId == nameOrItemId) {
                field = item;
                return false;
            }
        };
        this.cascade(fn);
        return field;
    }
    /**
     * Copies the current value to the reset value. Typically happens when this component was added to a parent and
     * when the form it belongs too loads.
     */
    trackReset() {
        this.findFields().forEach((field) => {
            field.trackReset();
        });
    }
    isModified() {
        const f = this.findFields();
        for (let i = 0, l = f.length; i < l; i++) {
            if (f[i].isModified()) {
                return true;
            }
        }
        return false;
    }
    internalSetValue(v) {
        for (let name in v) {
            // We cast to any[] for Ext compatibility. We try setValue() for Ext if it exists
            let fields = this.findFields(name);
            fields.forEach(field => field.setValue ? field.setValue(v[name]) : field.value = v[name]);
        }
    }
    internalGetValue() {
        // we have to clone the value to avoid side effects when the value is modified outside the form's
        // scope. We don't want any external modifications to leak in here because reference is a value.
        const formProps = structuredClone(this._value) || {};
        this.findFields().forEach((field) => {
            //for Extjs compat try .getName() and .getValue()
            const fieldName = (field.getName ? field.getName() : field.name);
            const fieldVal = field.getValue ? field.getValue() : field.value;
            if (fieldName) {
                if (field.disabled) {
                    delete formProps[fieldName];
                }
                else {
                    formProps[fieldName] = fieldVal;
                }
            }
        });
        return formProps;
    }
    validate() {
        super.validate();
        let invalid;
        this.findFields().forEach((i) => {
            if (!i.disabled && !i.isValid()) {
                invalid = i;
            }
        });
        if (invalid) {
            this.setInvalid("There's an invalid field");
        }
    }
    /**
     * Find the first invalid field
     */
    findFirstInvalid() {
        const items = this.findFields();
        for (let i = 0, l = items.length; i < l; i++) {
            if (!items[i].disabled && !items[i].isValid()) {
                return items[i];
            }
        }
        return undefined;
    }
    renderControl() {
    }
    clearInvalid() {
        super.clearInvalid();
        const items = this.findFields();
        items.forEach((field) => {
            field.clearInvalid();
        });
    }
    applyInvalidMsg() {
        if (this.invalidMsg) {
            this.el.classList.add("invalid");
        }
        else {
            this.el.classList.remove("invalid");
        }
    }
    /**
     * @inheritDoc
     */
    focus(o) {
        const fields = this.findFields();
        if (fields.length) {
            fields[0].focus(o);
            this.fire("focus", this, o);
        }
        else {
            super.focus(o);
        }
    }
}
/**
 * Shorthand function to create {@see ContainerField}
 *
 * @param config
 * @param items
 */
export const containerfield = (config, ...items) => createComponent(new ContainerField(), config, items);
//# sourceMappingURL=ContainerField.js.map