/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Observable } from "../Observable.js";
import { createComponent } from "../Component.js";
import { Format } from "../../util";
import { checkbox } from "../form";
import { btn } from "../Button";
import { menu } from "../menu";
export class TableColumn extends Observable {
    /**
     *
     * The column ID. Also used for 'property'
     */
    constructor(id) {
        super();
        this.id = id;
        /**
         * Make the column resizable by the user
         */
        this.resizable = false;
        /**
         * Make it sortable by the user
         */
        this.sortable = false;
        /**
         * Text alignment
         */
        this.align = "left";
        /**
         * Hide the column. It can be enabled by the user via the context menu.
         */
        this.hidden = false;
        /**
         * Enable this column in the enabled columns menu
         */
        this.hidable = true;
        this.property = id;
    }
}
/**
 * Create a table column
 *
 * @param config
 */
export const column = (config) => createComponent(new TableColumn(config.id), config);
export class DateTimeColumn extends TableColumn {
    constructor() {
        super(...arguments);
        this.renderer = (date) => {
            return Format.dateTime(date);
        };
        //argh!? https://stackoverflow.com/questions/43121661/typescript-type-inference-issue-with-string-literal
        this.align = "right";
        this.width = 190;
    }
}
/**
 * Create a column showing date and time
 * @param config
 */
export const datetimecolumn = (config) => createComponent(new DateTimeColumn(config.id), config);
export class DateColumn extends TableColumn {
    constructor() {
        super(...arguments);
        this.renderer = (date) => {
            return Format.date(date);
        };
        //argh!? https://stackoverflow.com/questions/43121661/typescript-type-inference-issue-with-string-literal
        this.align = "right";
        this.width = 128;
    }
}
/**
 * Create a column showing just a date
 *
 * @param config
 */
export const datecolumn = (config) => createComponent(new DateColumn(config.id), config);
export class CheckboxColumn extends TableColumn {
    constructor(id) {
        super(id);
        this.renderer = (val) => {
            return checkbox({
                value: val
            });
        };
        this.cls = "checkbox-select-column";
    }
}
/**
 * Create a checkbox column
 *
 * @param config
 */
export const checkboxcolumn = (config) => createComponent(new CheckboxColumn(config.id), config);
export class CheckboxSelectColumn extends TableColumn {
    constructor(id = "checkboxselect") {
        super(id);
        this.headerRenderer = (col, headerEl, table) => {
            return checkbox({
                listeners: {
                    change: (field, newValue, oldValue) => {
                        if (newValue) {
                            table.rowSelection.selectAll();
                        }
                        else {
                            table.rowSelection.clear();
                        }
                    }
                }
            });
        };
        this.renderer = (val, record, td, table, rowIndex) => {
            // add to selection model if value is true
            if (val && table.rowSelection) {
                table.rowSelection.add(rowIndex);
            }
            return checkbox({
                value: val,
                listeners: {
                    render: (field) => {
                        field.el.addEventListener("mousedown", (ev) => {
                            ev.stopPropagation();
                        });
                        table.rowSelection.on("selectionchange", () => {
                            field.value = table.rowSelection.selected.indexOf(rowIndex) > -1;
                        });
                    },
                    change: (field, newValue, oldValue) => {
                        const index = table.store.indexOf(record);
                        if (newValue) {
                            table.rowSelection.add(index);
                        }
                        else {
                            table.rowSelection.remove(index);
                        }
                    }
                }
            });
        };
        this.hidable = false;
        this.cls = "checkbox-select-column";
    }
}
export const checkboxselectcolumn = (config) => createComponent(new CheckboxSelectColumn(config && config.id ? config.id : "checkboxselect"), config);
/**
 * Creates a menu button.
 *
 * All items will have a property dataSet.rowIndex and dataSet.table so you know which record has been clicked on.
 *
 * @param items
 */
export const menucolumn = (...items) => column({
    width: 48,
    id: "btn",
    // headerRenderer: (col: TableColumn, headerEl: HTMLTableCellElement, table: Table) => {
    // 	headerEl.style.position = "sticky";
    // 	headerEl.style.right = "0";
    // 	return "";
    // },
    renderer: (columnValue, record, td, table, rowIndex) => {
        // td.style.position = "sticky";
        // td.style.right = "0";
        items.forEach(i => {
            i.dataSet.table = table;
            i.dataSet.rowIndex = rowIndex;
        });
        return btn({
            icon: "more_vert",
            menu: menu({}, ...items)
        });
    }
});
//# sourceMappingURL=TableColumns.js.map