/**
 * A javascript module to handle list items drag and drop
 *
 * Example of usage:
 *
 * Create a list (for example `<ul>` or `<tbody>`) where each draggable element has a drag handle.
 * The best practice is to use the template core/drag_handle:
 * $OUTPUT->render_from_template('core/drag_handle', ['movetitle' => get_string('movecontent', 'moodle', ELEMENTNAME)]);
 *
 * Attach this JS module to this list:
 *
 * Space between define and ( critical in comment but not allowed in code in order to function
 * correctly with Moodle's requirejs.php
 *
 * More details: https://docs.moodle.org/dev/Sortable_list
 *
 * For the full list of possible parameters see var defaultParameters below.
 *
 * The following jQuery events are fired:
 * - SortableList.EVENTS.DRAGSTART : when user started dragging a list element
 * - SortableList.EVENTS.DRAG : when user dragged a list element to a new position
 * - SortableList.EVENTS.DROP : when user dropped a list element
 * - SortableList.EVENTS.DROPEND : when user finished dragging - either fired right after dropping or
 *                          if "Esc" was pressed during dragging
 *
 * @example
 * define (['jquery', 'core/sortable_list'], function($, SortableList) {
 *     var list = new SortableList('ul.my-awesome-list'); // source list (usually <ul> or <tbody>) - selector or element
 *
 *     // Listen to the events when element is dragged.
 *     $('ul.my-awesome-list > *').on(SortableList.EVENTS.DROP, function(evt, info) {
 *         console.log(info);
 *     });
 *
 *     // Advanced usage. Overwrite methods getElementName, getDestinationName, moveDialogueTitle, for example:
 *     list.getElementName = function(element) {
 *         return $.Deferred().resolve(element.attr('data-name'));
 *     }
 * });
 *
 * @module     core/sortable_list
 * @class      core/sortable_list
 * @copyright  2018 Marina Glancy
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
define("core/sortable_list",["jquery","core/log","core/autoscroll","core/str","core/modal_factory","core/modal_events","core/notification"],(function($,log,autoScroll,str,ModalFactory,ModalEvents,Notification){var defaultParameters={targetListSelector:null,moveHandlerSelector:"[data-drag-type=move]",isHorizontal:!1,autoScroll:!0},CSS_keyboardDragClass="dragdrop-keyboard-drag",CSS_isDraggedClass="sortable-list-is-dragged",CSS_currentPositionClass="sortable-list-current-position",CSS_targetListClass="sortable-list-target",CSS_overElementClass="sortable-list-over-element",registerNotPassiveListeners=function(eventname){return{setup:function(x,ns,handle){return!!ns.includes("notPassive")&&(this.addEventListener(eventname,handle,{passive:!1}),!0)}}};(function(){var options,passivesupported=!1;try{options=Object.defineProperty({},"passive",{get:function(){passivesupported=!0}}),document.addEventListener("testpassiveeventoptions",options,options),document.removeEventListener("testpassiveeventoptions",options,options)}catch(err){passivesupported=!1}return passivesupported})&&($.event.special.touchstart=registerNotPassiveListeners("touchstart"),$.event.special.touchmove=registerNotPassiveListeners("touchmove"),$.event.special.touchend=registerNotPassiveListeners("touchend"));var SortableList=function(root,config){this.info=null,this.proxy=null,this.proxyDelta=null,this.dragCounter=0,this.lastEvent=null,this.config=$.extend({},defaultParameters,config||{}),this.config.listSelector=root,this.config.targetListSelector||(this.config.targetListSelector=root),"object"==typeof this.config.listSelector?$(this.config.listSelector).on("mousedown touchstart.notPassive",$.proxy(this.dragStartHandler,this)):$("body").on("mousedown touchstart.notPassive",this.config.listSelector,$.proxy(this.dragStartHandler,this)),null!==this.config.moveHandlerSelector&&$("body").on("click keypress",this.config.moveHandlerSelector,$.proxy(this.clickHandler,this))};return SortableList.EVENTS={DRAGSTART:"sortablelist-dragstart",DRAG:"sortablelist-drag",DROP:"sortablelist-drop",DRAGEND:"sortablelist-dragend"},SortableList.prototype.resetDraggedClasses=function(){var classes=[CSS_isDraggedClass,CSS_currentPositionClass,CSS_overElementClass,CSS_targetListClass];for(var i in classes)$("."+classes[i]).removeClass(classes[i]);this.proxy&&(this.proxy.remove(),this.proxy=$())},SortableList.prototype.calculatePositionOnPage=function(evt){if(evt.originalEvent&&evt.originalEvent.touches&&void 0!==evt.originalEvent.touches[0]){var touch=evt.originalEvent.touches[0];evt.pageX=touch.pageX,evt.pageY=touch.pageY}void 0===evt.pageX?(evt.pageX=this.lastEvent.pageX,evt.pageY=this.lastEvent.pageY):this.lastEvent=evt,void 0===evt.clientX&&(evt.clientX=Math.round(evt.pageX-$(window).scrollLeft()),evt.clientY=Math.round(evt.pageY-$(window).scrollTop()))},SortableList.prototype.dragStartHandler=function(evt){if(null!==this.info){if("click"===this.info.type||"touchend"===this.info.type)return;this.moveElement(this.info.sourceList,this.info.sourceNextElement),this.finishDragging()}if("mousedown"!==evt.type||1===evt.which){this.calculatePositionOnPage(evt);var movedElement=$(evt.target).closest($(evt.currentTarget).children());if(movedElement.length&&(null===this.config.moveHandlerSelector||$(evt.target).closest(this.config.moveHandlerSelector,movedElement).length)){evt.stopPropagation(),evt.preventDefault(),this.dragCounter++,this.info={element:movedElement,sourceNextElement:movedElement.next(),sourceList:movedElement.parent(),targetNextElement:movedElement.next(),targetList:movedElement.parent(),type:evt.type,dropped:!1,startX:evt.pageX,startY:evt.pageY,startTime:(new Date).getTime()},$(this.config.targetListSelector).addClass(CSS_targetListClass);var offset=movedElement.offset();movedElement.addClass(CSS_currentPositionClass),this.proxyDelta={x:offset.left-evt.pageX,y:offset.top-evt.pageY},this.proxy=$();var thisDragCounter=this.dragCounter;setTimeout($.proxy((function(){null!==this.info&&"click"!==this.info.type&&"keypress"!==this.info.type&&this.dragCounter===thisDragCounter&&this.createProxy()}),this),500),$(window).on("mousemove touchmove.notPassive mouseup touchend.notPassive",$.proxy(this.dragHandler,this)),$(window).on("keypress",$.proxy(this.dragcancelHandler,this)),this.config.autoScroll&&autoScroll.start((function(){$(window).trigger("mousemove")})),this.executeCallback(SortableList.EVENTS.DRAGSTART)}}},SortableList.prototype.createProxy=function(){this.proxy=this.info.element.clone(),this.info.sourceList.append(this.proxy),this.proxy.removeAttr("id").removeClass(CSS_currentPositionClass).addClass(CSS_isDraggedClass).css({position:"fixed"}),this.proxy.offset({top:this.proxyDelta.y+this.lastEvent.pageY,left:this.proxyDelta.x+this.lastEvent.pageX})},SortableList.prototype.clickHandler=function(evt){if(("keypress"!==evt.type||13===evt.originalEvent.keyCode||32===evt.originalEvent.keyCode)&&null===this.info){var clickedElement=$(evt.target).closest(this.config.moveHandlerSelector),sourceList=clickedElement.closest(this.config.listSelector),movedElement=clickedElement.closest(sourceList.children());movedElement.length&&(evt.preventDefault(),evt.stopPropagation(),this.dragCounter++,this.info={element:movedElement,sourceNextElement:movedElement.next(),sourceList:sourceList,targetNextElement:movedElement.next(),targetList:sourceList,dropped:!1,type:evt.type,startTime:(new Date).getTime()},this.executeCallback(SortableList.EVENTS.DRAGSTART),this.displayMoveDialogue(clickedElement))}},SortableList.prototype.getPositionInNode=function(pageX,pageY,element){if(!element.length)return null;var rect=element[0].getBoundingClientRect(),y=pageY-(rect.top+window.scrollY),x=pageX-(rect.left+window.scrollX);return x>=-0&&x<=rect.width+0&&y>=-0&&y<=rect.height+0?{x:x,y:y,xRatio:rect.width?x/rect.width:0,yRatio:rect.height?y/rect.height:0}:null},SortableList.prototype.isListHorizontal=function(element){var isHorizontal=this.config.isHorizontal;return!0===isHorizontal||!1===isHorizontal?isHorizontal:isHorizontal(element)},SortableList.prototype.dragHandler=function(evt){evt.preventDefault(),evt.stopPropagation(),this.calculatePositionOnPage(evt),this.proxy.offset({top:-1e3,left:-1e3});var element=$(document.elementFromPoint(evt.clientX,evt.clientY)),mainElement=this.info.element[0],isNotSelf=function(){return this!==mainElement},current=element.closest("."+CSS_targetListClass+" > :not(."+CSS_isDraggedClass+")").filter(isNotSelf),currentList=element.closest("."+CSS_targetListClass),proxy=this.proxy,isNotProxy=function(){return!proxy||!proxy.length||this!==proxy[0]};if($("."+CSS_overElementClass).removeClass(CSS_overElementClass),current.addClass(CSS_overElementClass),this.proxy.offset({top:this.proxyDelta.y+evt.pageY,left:this.proxyDelta.x+evt.pageX}),currentList.length&&!currentList.children().filter(isNotProxy).length)this.moveElement(currentList,$());else if(1===current.length&&!this.info.element.find(current[0]).length){var coordinates=this.getPositionInNode(evt.pageX,evt.pageY,current);if(coordinates){var parent=current.parent(),ratio=this.isListHorizontal(parent)?coordinates.xRatio:coordinates.yRatio,subList=current.find("."+CSS_targetListClass),subListEmpty=!subList.children().filter(isNotProxy).filter(isNotSelf).length;subList.length&&subListEmpty&&ratio>.2&&ratio<.8?this.moveElement(subList,$()):ratio>.5?this.moveElement(parent,current.next().filter(isNotProxy)):this.moveElement(parent,current)}}if("mouseup"===evt.type||"touchend"===evt.type){this.info.endX=evt.pageX,this.info.endY=evt.pageY,this.info.endTime=(new Date).getTime(),this.info.dropped=!0,this.info.positionChanged=this.hasPositionChanged(this.info);var oldinfo=this.info;this.executeCallback(SortableList.EVENTS.DROP),this.finishDragging(),"touchend"===evt.type&&null!==this.config.moveHandlerSelector&&oldinfo.endTime-oldinfo.startTime<500&&!oldinfo.positionChanged&&this.clickHandler(evt)}},SortableList.prototype.hasPositionChanged=function(info){return info.sourceList[0]!==info.targetList[0]||info.sourceNextElement.length!==info.targetNextElement.length||info.sourceNextElement.length&&info.sourceNextElement[0]!==info.targetNextElement[0]},SortableList.prototype.moveElement=function(parentElement,beforeElement){var dragEl=this.info.element;beforeElement.length&&beforeElement[0]===dragEl[0]||parentElement[0]===this.info.targetList[0]&&beforeElement.length===this.info.targetNextElement.length&&beforeElement[0]===this.info.targetNextElement[0]||(beforeElement.length?parentElement[0].insertBefore(dragEl[0],beforeElement[0]):this.proxy&&this.proxy.parent().length&&this.proxy.parent()[0]===parentElement[0]?parentElement[0].insertBefore(dragEl[0],this.proxy[0]):parentElement[0].appendChild(dragEl[0]),this.info.targetList=parentElement,this.info.targetNextElement=beforeElement,this.executeCallback(SortableList.EVENTS.DRAG))},SortableList.prototype.finishDragging=function(){this.resetDraggedClasses(),this.config.autoScroll&&autoScroll.stop(),$(window).off("mousemove touchmove.notPassive mouseup touchend.notPassive",$.proxy(this.dragHandler,this)),$(window).off("keypress",$.proxy(this.dragcancelHandler,this)),this.executeCallback(SortableList.EVENTS.DRAGEND),this.info=null},SortableList.prototype.executeCallback=function(eventName){this.info.element.trigger(eventName,this.info)},SortableList.prototype.dragcancelHandler=function(evt){"keypress"===evt.type&&27===evt.originalEvent.keyCode&&(this.moveElement(this.info.sourceList,this.info.sourceNextElement),this.finishDragging())},SortableList.prototype.getElementName=function(element){return $.Deferred().resolve(element.text())},SortableList.prototype.getDestinationName=function(parentElement,afterElement){return afterElement.length?this.getElementName(afterElement).then((function(name){return str.get_string("movecontentafter","moodle",name)})):str.get_string("movecontenttothetop","moodle")},SortableList.prototype.getMoveDialogueTitle=function(element,handler){return handler.attr("title")?$.Deferred().resolve(handler.attr("title")):this.getElementName(element).then((function(name){return str.get_string("movecontent","moodle",name)}))},SortableList.prototype.getDestinationsList=function(){var addedLists=[],targets=$(this.config.targetListSelector),destinations=$("<ul/>").addClass(CSS_keyboardDragClass),result=$.when().then((function(){return destinations})),createLink=$.proxy((function(parentElement,beforeElement,afterElement){beforeElement.is(this.info.element)||afterElement.is(this.info.element)||$.contains(this.info.element[0],parentElement[0])||(result=result.then($.proxy((function(){return this.getDestinationName(parentElement,afterElement)}),this)).then((function(txt){var li=$("<li/>").appendTo(destinations);return $('<a href="#"/>').attr("data-core_sortable_list-quickmove",1).appendTo(li).data("parent-element",parentElement).data("before-element",beforeElement).text(txt),destinations})))}),this),addList=function(){if(-1===$.inArray(this,addedLists)){addedLists.push(this);var list=$(this),children=list.children();children.each((function(){var element=$(this);createLink(list,element,element.prev()),element.find(targets).each(addList)})),createLink(list,$(),children.last())}};return targets.each(addList),result},SortableList.prototype.displayMoveDialogue=function(clickedElement){ModalFactory.create({type:ModalFactory.types.CANCEL,title:this.getMoveDialogueTitle(this.info.element,clickedElement),body:this.getDestinationsList()}).then($.proxy((function(modal){var quickMoveHandler=$.proxy((function(e){e.preventDefault(),e.stopPropagation(),this.moveElement($(e.currentTarget).data("parent-element"),$(e.currentTarget).data("before-element")),this.info.endTime=(new Date).getTime(),this.info.positionChanged=this.hasPositionChanged(this.info),this.info.dropped=!0,clickedElement.focus(),this.executeCallback(SortableList.EVENTS.DROP),modal.hide()}),this);return modal.getRoot().on("click","[data-core_sortable_list-quickmove]",quickMoveHandler),modal.getRoot().on(ModalEvents.hidden,$.proxy((function(){modal.getRoot().off("click","[data-core_sortable_list-quickmove]",quickMoveHandler),modal.destroy(),this.finishDragging()}),this)),modal.setLarge(),modal.show(),modal}),this)).catch(Notification.exception)},SortableList}));

//# sourceMappingURL=sortable_list.min.js.map