Internationalization
====================

System Languages
----------------
At various parts of the system, there are several possibilities for the
language selection of the "local" language:

  1. User (client or agent) preference
  2. Ticket thread (for system activity notes)
  3. System (for logs and administrative messages)

The system is flexible enough to support these different cases and provides
a few wrapper functions to connect your string to the appropriate language.
Bear in mind when writing new code that strings may need to be translated
into more than one language. For instance, if a string is to be displayed as
an error in a web page as well as appear in an email to the administrator,
it may need to be translated differently for both.

Consider a Spanish-speaking user visiting a German-based help desk. After
attempting to log in several times, they receive an error banner with a
particular message. A message is also sent to the site administrator warning
about possible brute force attack. These messages will need to consider
different audiences when being localized. The site administrator should
receive a warning email in German; whereas the user should see a Spanish
error message with details about what to do next.

Creating localized strings
--------------------------
Creating localized strings in osTicket is similar to creating localized
strings in any gettext-based project. osTicket has opted to use a pure-php
version of gettext in order to avoid possible pitfalls surrounding usage of
gettext with PHP including

  * MO file caching requiring HTTP server restart
  * `gettext` missing from the PHP installation
  * Requirement of locale pre-configuration on the server

### Adding new strings

Use a few function calls to get your text localized:

  * `__('string')` localize the string to the current user preference
  * `_N('string', 'strings', n)` localize a string with plural alternatives
    to the current user locale preference.
  * `_S('string')` localize the string to the system primary language
  * `_NS('string', 'strings', n)` localize a string with plural alternatives
    to the system primary language.
  * `_L('string', locale)` localize a string to a named locale. This is
    useful in a ticket thread, for instance, which may have a language
    preference separate from both the user and the system primary language
  * `_NL('string', 'strings', n, locale)` localize a string with plural
    alternatives to a specific locale.
  * `_P('context', 'string')` localize a string which may have the same
    source text as another string but have a different context or usage. An
    example would be `open` used as a noun, adjective, and verb.
  * `_NP('context', 'string', 'strings', n)` localize a string with plural
    alternatives which may have different contexts in the source language.

In some cases, it is not possible to use a function to translate your
string. For instance, if it is used a as a class constant or default value
for a class variable. In such a case, a hint can be used to tell the POT
scanner to translate the string for use elsewhere. As an example, one might
set something like this up:

```php
    class A {
        static $name = /* @trans */ 'Localized string';
    }

    print __(A::$name);
```

In this case the localized version of the class variable is translated when
it is used — not when it is defined.

The `* @trans *` text in comment is significant. Both the asterisks and the
term `@trans` are used to signify that the phrase should be made
translatable. The comment must also be placed immedately before or after the
string without any other PHP symbols in between (like the semi-colon).

### Adding context to your strings

Your text may be ambiguous or unclear when it is viewed outside the context
of the code in which it appears. The system allows adding of comments
similar to the stock gettext tools. Any comments written directly beside
(behind or in front of) a localized string will be captured with the string
in the translation template. For instance

```php
    print __('Localized' /* These comments are exported */);
```

All types of PHP comments are supported. They must be placed inside the
parentheses of the localization call. If using single-line comments, use
multiple lines if necessary to define the call so that your single-line
comments can be used. For instance:

```php
    print sprintf(__(
        // Tokens will be <a> of <b> <object(s)>
        '%1$d of %2$d %3$s'
        ),
        $a, $b,
        _N('object', 'objects', $b)
    );
```

### Building POT file for translations

Use the command line to compile the POT file to standard out

    php manage.php i18n make-pot > message.pot

### Building language packs

In an effort for the php version of gettext to offer similar performance to
the extension counterpart, a variant of the MO file is used which is a PHP
serialized array written to a file. The original MO file functions basically
like a text array. In stead of searching through the MO file for each string
to be translated, the original and translated texts are placed into a hash
array for quick access and the hash array is serialized for the language
pack. At runtime, the hash array is recreated from the export and the
strings are quickly accessed from the PHP hash array.

A MO file can be manually compiled using a command-line interface

    php include/class.translation.php message.mo > message.mo.php
